# CSG-EB

This library is for parsing [Constructive Solid Geometry
(CSG)](https://github.com/openscad/openscad/wiki/CSG-File-Format) for use in
[AMReX Embedded Boundaries
(EB)](https://amrex-codes.github.io/amrex/docs_html/EB.html).

CSG is a file format used by the open source CAD modelling tool
[OpenSCAD](https://www.openscad.org). CSG is a subset of the SCAD language,
equivalent in the kinds of geometries that can be expressed, but with fewer
primitives (OpenSCAD can export `.scad` files to `.csg`). SCAD code is easier to
read and write by humans; CSG files are easier to parse and process by software.

Dependencies:

 - [PEGTL](https://github.com/taocpp/PEGTL) for parsing
 - [Catch2](https://github.com/catchorg/Catch2) for testing framework
 - [CGAL](https://www.cgal.org) for geometry computation



## How to Build

### Build dependencies

 - C++17 compiler (GCC >=7.x, Clang >=5.x)
 - CMake >=3.14
 - Conan (CMake will try to install Conan if `conan` is not in `PATH`)

CMake will try to install [Conan](https://conan.io) if `conan` is not in `PATH`.
However, installing Conan (`pip install conan`) is recommended for faster builds.

#### Build

```shell
> cmake -S . -B build -DCMAKE_BUILD_TYPE=Release
> cmake --build build
```

#### Run tests

```shell
> cd build
> ctest
```


## Using CSG-EB in your project

 - Include the header file `csg.hpp`
 - Call `csg::get_csgif` with the path to a CSG file to create an [Implicit Function object](https://amrex-codes.github.io/amrex/docs_html/EB.html#implicit-function).

```cpp
#include <csg.hpp>

// for EB2::makeShop
#include <AMReX_EB2_GeometryShop.H>

...


auto csg_file = "~/my_geometry.csg";
auto is_internal_flow = true;

auto csg_if = csg::get_csgif(csg_file, is_internal_flow);

auto gshop = EB2::makeShop(*csg_if);

```


### Note on the return value

If `is_internal_flow==true`, the CSG geometry defines a hollow space for the domain:

   - csg_if(x, y, z) < 0 ⇒ (x, y, z) is **INSIDE**
   - csg_if(x, y, z) > 0 ⇒ (x, y, z) is **OUTSIDE**

If `is_internal_flow==false`, the CSG geometry defines a solid boundary to the domain:

   - csg_if(x, y, z) < 0 ⇒ (x, y, z) is **OUTSIDE**
   - csg_if(x, y, z) > 0 ⇒ (x, y, z) is **INSIDE**


### Add to the CMake build

* Add this repo as a CMake subdirectory in your `CMakeLists.txt`:

```cmake
add_subdirectory(/path/to/repo/csg-eb)
```

* Add the `csg` target to the CMake target that uses code like in the the above example:

```cmake
target_link_libraries(<target> PRIVATE csg)
```
