// standard includes
#include <assert.h>
#include <iostream>
#include <memory>

// subproject includes
#include <tao/pegtl.hpp>

// includes
#include "solver.hpp"

using namespace tao::pegtl;

namespace {

struct parser_state {
  std::string current_key;
  std::variant<std::vector<double>, std::vector<std::string>> current_value;
  std::vector<double> current_num_vec;
  std::vector<std::string> current_str_vec;
  solver::InputInfo info;
};

} // namespace

namespace {

struct escaped_x : seq<one<'x'>, rep<2, must<xdigit>>> {};
struct escaped_u : seq<one<'u'>, rep<4, must<xdigit>>> {};
struct escaped_U : seq<one<'U'>, rep<8, must<xdigit>>> {};
struct escaped_c
    : one<'\'', '"', '?', '\\', 'a', 'b', 'f', 'n', 'r', 't', 'v'> {};

struct escaped : sor<escaped_x, escaped_u, escaped_U, escaped_c> {};

struct key : plus<sor<plus<alnum>, one<'_'>, one<'-'>, one<'.'>>> {};

struct character
    : if_must_else<one<'\\'>, escaped, utf8::range<0x20, 0x10FFFF>> {};
struct quoted_string : if_must<one<'"'>, until<one<'"'>, character>> {};
struct unquoted_string : key {};
struct string_literal : sor<quoted_string, unquoted_string> {};

struct plus_minus : opt<one<'+', '-'>> {};
struct dot : one<'.'> {};

struct decimal
    : if_then_else<dot, plus<digit>, seq<plus<digit>, opt<dot, star<digit>>>> {
};
struct e : one<'e', 'E'> {};
struct p : one<'p', 'P'> {};
struct exponent : seq<plus_minus, plus<digit>> {};
struct double_ : seq<plus_minus, decimal, opt<e, exponent>> {};

struct num_array : list<double_, plus<blank>> {};
struct str_array : list<string_literal, plus<blank>> {};

struct value : sor<num_array, str_array> {};
struct comment : if_must<one<'#'>, until<eolf>> {};

struct value_item : pad<value, blank> {};
struct value_list : list_must<value_item, one<','>> {};
struct keyval_line
    : if_must<pad<key, space>, string<'='>, value_list, sor<comment, eolf>> {};
struct comment_line : seq<one<'#'>, until<eolf>> {};
struct blank_line : seq<star<blank>, until<eolf>> {};
struct line : sor<comment_line, keyval_line, blank_line> {};
struct grammar : until<eof, line> {};

template <typename Rule> struct action {};

template <> struct action<key> {
  template <typename Input>
  static void apply(const Input &in, parser_state &st) {
    std::stringstream ss(in.string());
    ss >> st.current_key;
  }
};

template <> struct action<quoted_string> {
  template <typename Input>
  static void apply(const Input &in, parser_state &st) {
    std::stringstream ss(in.string());
    std::string v;
    ss >> v;
    auto str_val = v.substr(1, v.length() - 2); // strip quotes
    st.current_str_vec.push_back(str_val);
  }
};

template <> struct action<unquoted_string> {
  template <typename Input>
  static void apply(const Input &in, parser_state &st) {
    std::stringstream ss(in.string());
    std::string v;
    ss >> v;
    st.current_str_vec.push_back(v);
  }
};

template <> struct action<double_> {
  template <typename Input>
  static void apply(const Input &in, parser_state &st) {
    std::stringstream ss(in.string());
    double v;
    ss >> v;
    st.current_num_vec.push_back(v);
  }
};

template <> struct action<num_array> {
  template <typename Input>
  static void apply(const Input &in, parser_state &st) {
    std::stringstream ss(in.string());
    st.current_value = st.current_num_vec;
    st.current_num_vec.clear();
  }
};

template <> struct action<str_array> {
  template <typename Input>
  static void apply(const Input &in, parser_state &st) {
    std::stringstream ss(in.string());
    st.current_value = st.current_str_vec;
    st.current_str_vec.clear();
  }
};

template <> struct action<keyval_line> {
  template <typename Input>
  static void apply(const Input &in, parser_state &st) {
    std::stringstream ss(in.string());
    st.info[st.current_key] = st.current_value;
  }
};

std::optional<parser_state> do_parse(std::string str) {
  parser_state st;
  memory_input in(str, "std::cin");

  try {
    if (!parse<grammar, action>(in, st)) {
      return std::nullopt;
    }
  } catch (const parse_error &e) {
    const auto p = e.positions.front();
    std::cerr << e.what() << std::endl
              << in.line_at(p) << std::endl
              << std::string(p.byte_in_line, ' ') << '^' << std::endl;
  }

  return st;
}

} // namespace

namespace solver {

std::optional<solver::InputInfo> parse_inputs(std::string str) {
  auto maybe_state = do_parse(str);
  if (!maybe_state.has_value()) {
    return std::nullopt;
  }
  return maybe_state.value().info;
}
} // namespace solver
