#ifndef SOLVER_H_
#define SOLVER_H_

#include <array>
#include <map>
#include <memory>
#include <optional>
#include <string>
#include <variant>
#include <vector>

#include <inputs.hpp>

namespace solver {

using NumberArray = std::vector<double>;
using StringArray = std::vector<std::string>;
using InputInfo = std::map<std::string, std::variant<NumberArray, StringArray>>;

std::optional<InputInfo> parse_inputs(std::string);

struct GeometryAxis {
  double low;
  double high;
  bool periodic;
};
struct GeometrySettings {
  std::array<GeometryAxis, 3> axes;
  std::string csg_filename;
};

struct MeshAxis {
  unsigned int fluid_max_tile_size;
  unsigned int max_grid_size;
  unsigned int n_cell;
  unsigned int particle_max_grid_size;
  unsigned int particle_max_tile_size;
};

struct MeshSettings {
  int blocking_factor;
  double small_volfrac;
  std::array<MeshAxis, 3> axes;
};
struct TimeSettings {
  bool fixed_dt;
  double cfl;
  double dt_max;
  double dt_min;
  double tcoll_ratio;
  double tstop;
  unsigned int max_step;
};
struct SolverSettings {
  GeometrySettings geometry;
  MeshSettings mesh;
  TimeSettings time;
};

struct InputsMessage {
  std::string message;
};
std::pair<solver::SolverSettings, std::vector<InputsMessage>>
    make_solver(solver::InputInfo);
std::string serialize(solver::SolverSettings);

} // namespace solver

#endif
