#ifndef CSG_H_
#define CSG_H_

#if defined(CSG_USE_GPU) && !defined(CSG_GPU_HOST_DEVICE)
#define CSG_GPU_HOST_DEVICE __host__ __device__
#else
#define CSG_GPU_HOST_DEVICE
#endif

#include <array>
#include <memory>
#include <tuple>
#include <vector>

namespace csg {

struct GPUable {};
template <class D, class Enable = void> struct IsGPUable : std::false_type {};
template <class D>
struct IsGPUable<
    D, typename std::enable_if<std::is_base_of<GPUable, D>::value>::type>
    : std::true_type {};

struct Tree;
std::shared_ptr<Tree> parse_csg(std::string);

class CsgIF : GPUable {
public:
  CsgIF(std::shared_ptr<Tree> a_state);
  CsgIF(const CsgIF &rhs);
  CsgIF(CsgIF &&rhs) noexcept;
  CsgIF &operator=(const CsgIF &rhs) = delete;
  CsgIF &operator=(CsgIF &&rhs) = delete;
  ~CsgIF();

  CSG_GPU_HOST_DEVICE
  double operator()(double xx, double yy, double zz) const noexcept;

  inline double operator()(const std::array<double, 3> &p) const noexcept {
    return this->operator()(p[0], p[1], p[2]);
  }

private:
  std::shared_ptr<Tree> m_state;

  class Impl;
  std::unique_ptr<Impl> m_pimpl;
};

std::shared_ptr<csg::Tree> get_csgtree_from_filename(std::string);
std::unique_ptr<CsgIF> get_csgif_from_filename(std::string geom_file);

} // namespace csg

#endif
