#ifndef CSG_TYPES_H_
#define CSG_TYPES_H_

#include <array>
#include <memory>
#include <optional>
#include <tuple>
#include <variant>
#include <vector>

namespace csg {

struct Circle {
  std::optional<std::string> name;
  double radius;
};

struct Square {
  std::optional<std::string> name;
  std::tuple<double, double> size;
  bool center;
};

struct Sphere {
  std::optional<std::string> name;
  double radius;
};

struct Cube {
  std::optional<std::string> name;
  std::tuple<double, double, double> size;
  bool center;
};

struct Cylinder {
  std::optional<std::string> name;
  double radius;
  double height;
  bool center;
};

struct Cone {
  std::optional<std::string> name;
  double radius1;
  double radius2;
  double height;
  bool center;
};

enum Dimension { D2, D3 };

template <Dimension dim> struct Mulmatrix;
template <Dimension dim> struct Union;
template <Dimension dim> struct Intersection;
template <Dimension dim> struct Difference;
struct LinearExtrude;
struct RotateExtrude;

template <Dimension dim> struct TypeHelper;

template <> struct TypeHelper<Dimension::D2> {
  using Type =
      std::variant<Circle, Square, Union<Dimension::D2>,
                   Intersection<Dimension::D2>, Difference<Dimension::D2>,
                   Mulmatrix<Dimension::D2>>;
};

template <> struct TypeHelper<Dimension::D3> {
  using Type =
      std::variant<Sphere, Cube, Cylinder, Cone, Union<Dimension::D3>,
                   Intersection<Dimension::D3>, Difference<Dimension::D3>,
                   Mulmatrix<Dimension::D3>, LinearExtrude, RotateExtrude>;
};

template <Dimension dim> struct Union {
  std::vector<typename TypeHelper<dim>::Type> objs;
};

template <Dimension dim> struct Intersection {
  std::vector<typename TypeHelper<dim>::Type> objs;
};

template <Dimension dim> struct Difference {
  std::shared_ptr<typename TypeHelper<dim>::Type> first_obj;
  Union<dim> next_objs;
};

template <> struct Mulmatrix<Dimension::D3> {
  std::array<std::array<double, 3>, 3> rotation;
  std::array<double, 3> translation;
  Union<Dimension::D3> group;
};

template <> struct Mulmatrix<Dimension::D2> {
  std::array<std::array<double, 2>, 2> rotation;
  std::array<double, 2> translation;
  Union<Dimension::D2> group;
};

struct LinearExtrude {
  double height;
  bool center;
  double twist;
  Union<Dimension::D2> group;
};

struct RotateExtrude {
  double angle;
  Union<Dimension::D2> group;
};

struct Tree {
  Union<Dimension::D3> top;
};

const double EXTERNAL_FLOW = -1.0;

double signed_distance_3d(const Union<Dimension::D3> &, double, double, double);

// defining some useful aliases
using Mulmatrix3D = Mulmatrix<Dimension::D3>;
using Mulmatrix2D = Mulmatrix<Dimension::D2>;
using Union3D = Union<Dimension::D3>;
using Union2D = Union<Dimension::D2>;
using Intersection3D = Intersection<Dimension::D3>;
using Intersection2D = Intersection<Dimension::D2>;
using Difference3D = Difference<Dimension::D3>;
using Difference2D = Difference<Dimension::D2>;
using Type2D = TypeHelper<Dimension::D2>::Type;
using Type3D = TypeHelper<Dimension::D3>::Type;

} // namespace csg

#endif
