#include "catch2/catch.hpp"

#include <csg.hpp>
#include <csg_types.hpp>
#include <iostream>

namespace {
TEST_CASE("linear extrude", "[csg]") {
  auto st = *csg::parse_csg(R"(
linear_extrude(
height = 10,
center = true,
twist = 0
) {
    circle(r = 1);
}
)");
  auto lin_ext = std::get<csg::LinearExtrude>(st.top.objs.back());
  CHECK(lin_ext.group.objs.size() == 1);
  CHECK(lin_ext.height == 10);
  CHECK(lin_ext.center == true);
  CHECK(lin_ext.twist == 0);

  auto cir = std::get<csg::Circle>(lin_ext.group.objs.at(0));
  CHECK(cir.radius == 1);
}

TEST_CASE("rotate extrude", "[csg]") {
  auto st = *csg::parse_csg(R"(
rotate_extrude(
angle = 90
) {
    square(size = [2, 3], center = false);
}
)");
  auto rot_ext = std::get<csg::RotateExtrude>(st.top.objs.back());
  CHECK(rot_ext.group.objs.size() == 1);
  CHECK(rot_ext.angle == 90);

  auto sq = std::get<csg::Square>(rot_ext.group.objs.at(0));
  auto [xx, yy] = sq.size;
  CHECK(xx == 2);
  CHECK(yy == 3);
}

TEST_CASE("extrude two shapes", "[csg]") {
  auto st = *csg::parse_csg(R"(
linear_extrude(
height = 10,
center = true,
twist = 0
) {
    circle(r = 1);
    square(size = [2, 2], center = false);
}
)");

  auto lin_ext = std::get<csg::LinearExtrude>(st.top.objs.back());
  CHECK(lin_ext.group.objs.size() == 2);
  CHECK(lin_ext.height == 10);
  CHECK(lin_ext.center == true);
  CHECK(lin_ext.twist == 0);

  auto cir = std::get<csg::Circle>(lin_ext.group.objs.at(0));
  CHECK(cir.radius == 1);

  auto sq = std::get<csg::Square>(lin_ext.group.objs.at(1));
  auto [xx, yy] = sq.size;
  CHECK(xx == 2);
  CHECK(yy == 2);
}

TEST_CASE("extrude with mulmatrix", "[csg]") {
  auto st = *csg::parse_csg(R"(
linear_extrude(height = 10, center = false, twist = 5) {
	circle(r = 2);
	multmatrix([[1, 0, 0, 5], [0, 1, 0, 5], [0, 0, 1, 0], [0, 0, 0, 1]]) {
		square(size = [4, 5], center = false);
	}
}
)");

  auto lin_ext = std::get<csg::LinearExtrude>(st.top.objs.back());
  CHECK(lin_ext.group.objs.size() == 2);
  CHECK(lin_ext.height == 10);
  CHECK(lin_ext.center == false);
  CHECK(lin_ext.twist == 5);

  auto cir = std::get<csg::Circle>(lin_ext.group.objs.at(0));
  CHECK(cir.radius == 2);

  auto mm = std::get<csg::Mulmatrix2D>(lin_ext.group.objs.at(1));
  auto [Cx, Cy] = mm.translation;
  CHECK(Cx == 5);
  CHECK(Cy == 5);
  CHECK(mm.group.objs.size() == 1);

  auto sq = std::get<csg::Square>(mm.group.objs.at(0));
  auto [xx, yy] = sq.size;
  CHECK(xx == 4);
  CHECK(yy == 5);
}

TEST_CASE("extrude with 3D object", "[csg]") {
  auto st = csg::parse_csg(R"(
linear_extrude(height = 10, center = false, twist = 5) {
	multmatrix([[1, 0, 0, 5], [0, 1, 0, 5], [0, 0, 1, 0], [0, 0, 0, 1]]) {
		sphere(r = 8);
	}
}
)");

  CHECK(st == nullptr);
}

} // namespace
