#include "catch2/catch.hpp"

#include <solver.hpp>

TEST_CASE("comment", "[]") {
  auto maybe_st = solver::parse_inputs(R"(
# this is a comment
)");
  CHECK(maybe_st.has_value());
}

TEST_CASE("integer", "[]") {
  auto maybe_st = solver::parse_inputs(R"(
mfix.max_step = 100
)");
  REQUIRE(maybe_st.has_value());
  auto info = maybe_st.value();
  auto value = info["mfix.max_step"];
  auto aa = std::get<solver::NumberArray>(value);
  CHECK(aa.size() == 1);
  CHECK(aa[0] == 100);
}

TEST_CASE("double", "[]") {
  auto maybe_st = solver::parse_inputs(R"(
mfix.stop_time = 0.20
)");
  REQUIRE(maybe_st.has_value());
  auto info = maybe_st.value();
  auto value = info["mfix.stop_time"];
  auto aa = std::get<solver::NumberArray>(value);
  CHECK(aa.size() == 1);
  CHECK(aa[0] == 0.2);
}

TEST_CASE("string", "[]") {
  auto maybe_st = solver::parse_inputs(R"(
  mfix.particle_init_type = "Auto"
)");
  REQUIRE(maybe_st.has_value());
  auto info = maybe_st.value();
  auto value = info["mfix.particle_init_type"];
  auto ss = std::get<solver::StringArray>(value);
  CHECK(ss.size() == 1);
  CHECK(ss[0] == "Auto");
}

TEST_CASE("implicit string", "[]") {
  auto maybe_st = solver::parse_inputs(R"(
fluid.solve = fluid
)");
  REQUIRE(maybe_st.has_value());
  auto info = maybe_st.value();
  auto value = info["fluid.solve"];
  auto ss = std::get<solver::StringArray>(value);
  CHECK(ss.size() == 1);
  CHECK(ss[0] == "fluid");
}

TEST_CASE("tabbed array", "[]") {
  auto maybe_st = solver::parse_inputs(R"(
 amr.avg_vel_p	=	0	0	0
)");

  REQUIRE(maybe_st.has_value());
  auto info = maybe_st.value();
  auto value = info["amr.avg_vel_p"];
  auto aa = std::get<solver::NumberArray>(value);
  CHECK(aa.size() == 3);
  CHECK(aa[0] == 0.0);
  CHECK(aa[1] == 0.0);
  CHECK(aa[2] == 0.0);
}

TEST_CASE("double array", "[]") {
  auto maybe_st = solver::parse_inputs(R"(
  mfix.gravity = -9.81  0.0  0.0
)");

  REQUIRE(maybe_st.has_value());
  auto info = maybe_st.value();
  auto value = info["mfix.gravity"];
  auto aa = std::get<solver::NumberArray>(value);
  CHECK(aa.size() == 3);
  CHECK(aa[0] == -9.81);
  CHECK(aa[1] == 0.0);
  CHECK(aa[2] == 0.0);
}

TEST_CASE("string array", "[]") {
  auto maybe_st = solver::parse_inputs(R"(
  mfix.regions = abc  ijk  xyz
)");

  REQUIRE(maybe_st.has_value());
  auto info = maybe_st.value();
  auto value = info["mfix.regions"];
  auto aa = std::get<solver::StringArray>(value);
  CHECK(aa.size() == 3);
  CHECK(aa[0] == "abc");
  CHECK(aa[1] == "ijk");
  CHECK(aa[2] == "xyz");
}

TEST_CASE("multiple arrays", "[]") {
  auto maybe_st = solver::parse_inputs(R"(
geometry.prob_lo     = -4      -1      -2
geometry.prob_hi     = 0.004   0.001   0.001
geometry.is_periodic = 1       0       1
)");

  REQUIRE(maybe_st.has_value());
  auto info = maybe_st.value();
  CHECK(info.count("geometry.prob_hi"));
  CHECK(info.count("geometry.prob_lo"));
  CHECK(info.count("geometry.is_periodic"));

  auto highs = info["geometry.prob_lo"];
  auto aa = std::get<solver::NumberArray>(highs);
  CHECK(aa.size() == 3);
  CHECK(aa[0] == -4);
  CHECK(aa[1] == -1);
  CHECK(aa[2] == -2);

  auto lows = info["geometry.prob_hi"];
  auto bb = std::get<solver::NumberArray>(lows);
  CHECK(bb.size() == 3);
  CHECK(bb[0] == 0.004);
  CHECK(bb[1] == 0.001);
  CHECK(bb[2] == 0.001);

  auto pers = info["geometry.is_periodic"];
  auto cc = std::get<solver::NumberArray>(pers);
  CHECK(cc.size() == 3);
  CHECK(cc[0] == 1);
  CHECK(cc[1] == 0);
  CHECK(cc[2] == 1);
}

TEST_CASE("whole_file", "[]") {
  auto maybe_st = solver::parse_inputs(R"(

#_______________________________________________________________________
# Solver settings

#amrex.fpe_trap_invalid = 1

#! Fluid solver
#!-----------------------------------------------------------//
mfix.cfl = 0.5

mfix.max_step = 100
#mfix.stop_time = 0.20

mfix.particle_init_type = "Auto"

mfix.gravity = -9.81  0.0  0.0

mfix.drag_type = "BVK2"

mac_proj.verbose = 0
nodal_proj.verbose = 1

amr.blocking_factor = 1

#_______________________________________________________________________
# Geometry / grids / tiles

# Maximum level in hierarchy (for now must be 0, i.e., one level in total)
amr.max_level = 0

geometry.coord_sys   = 0                       # 0: Cartesian
geometry.is_periodic = 0       0       0       # Is periodic in each direction?
geometry.prob_lo     = 0.      0.      0.      # lo corner of physical domain
geometry.prob_hi     = 0.004   0.001   0.001   # hi corner of physical domain

# Number of grid cells in each direction at the coarsest level
amr.n_cell =  20   5  5

#! Grids
#!-----------------------------------------------------------//
# Maximum allowable size of each fluid subdomain in the problem domain;

# Fluid
amr.max_grid_size_x = 1024
amr.max_grid_size_y = 1024
amr.max_grid_size_z = 1024

# Particles (not with KDTree)
#particles.max_grid_size_x =   32
#particles.max_grid_size_y =   32
#particles.max_grid_size_z =   32

#! Tiles
#!-----------------------------------------------------------//

# Fluid: Maximum tile size within each grid
fabarray.mfiter_tile_size = 1024 1024 1024

# Particles: Maximum particle title size
particles.tile_size = 1024 1024 1024


#! EB settings
#!-----------------------------------------------------------//

# Level-set refinement and padding
mfix.levelset__refinement    = 4 # levelset resolution

mfix.write_eb_surface = true  # Needed to output eb-surfaces


#_______________________________________________________________________
# Particle load balancing

#mfix.load_balance_type = "KnapSack"

amr.dual_grid          =  0
amr.regrid_int         = -1

#! KnapSack settings
#!-----------------------------------------------------------//
#default is "RunTimeCosts"; options include RunTimeCosts or NumParticles
#this option is only relevant if load_balance_type = KnapSack

#mfix.knapsack_weight_type = "NumParticles"
#mfix.knapsack_weight_type = "RunTimeCosts"

#_______________________________________________________________________
# IO / Checkpointing

amr.par_ascii_int  = -1
amr.par_ascii_file ="vis"

amr.plot_int       = -1
amr.plot_file      ="plt"

amr.check_int      = -1
amr.check_file     ="chk"

#! Restart from checkpoint
#!-----------------------------------------------------------//
#amr.restart   ="chk00100"


#_______________________________________________________________________
# Fluid model settings
#
fluid.solve = fluid

fluid.density            = constant
fluid.density.constant   = 1.0

fluid.viscosity          = constant
fluid.viscosity.constant = 2.0e-5


#_______________________________________________________________________
# DEM model settings
#
dem.solve = solid0

dem.friction_coeff.pp     =     0.0
dem.friction_coeff.pw     =     0.0

dem.spring_const.pp       =    10.0
dem.spring_const.pw       =    10.0

dem.restitution_coeff.solid0.solid0  =     0.8
dem.restitution_coeff.solid0.wall    =     1.0

dem.spring_tang_fac.pp    =     0.285714285
dem.spring_tang_fac.pw    =     0.285714285

dem.damping_tang_fac.pp   =     0.5
dem.damping_tang_fac.pw   =     0.5


#_______________________________________________________________________
# EB geometry
#
mfix.geometry = "cylinder"

cylinder.internal_flow = true

cylinder.radius = 0.00045
cylinder.height = -1.0

cylinder.direction = 0
cylinder.center    =  0.0020  0.0005   0.0005


#_______________________________________________________________________
# Regions for defining ICs and BCs
#
mfix.regions = full-domain  bed  inflow  outflow

regions.full-domain.lo = 0.0000  0.0000  0.0000
regions.full-domain.hi = 0.0040  0.0010  0.0010

regions.bed.lo         = 0.0003  0.0000  0.0000
regions.bed.hi         = 0.0032  0.0010  0.0010

regions.inflow.lo      = 0.0000  0.0000  0.0000
regions.inflow.hi      = 0.0000  0.0010  0.0010

regions.outflow.lo     = 0.0040  0.0000  0.0000
regions.outflow.hi     = 0.0040  0.0010  0.0010

#_______________________________________________________________________
# Initial Conditions
#
ic.regions = full-domain  bed

ic.full-domain.fluid.volfrac   = 1.0
ic.full-domain.fluid.velocity  = 0.015  0.0  0.0

ic.bed.fluid.volfrac   =  0.725
ic.bed.fluid.velocity  =  0.00  0.00  0.00

ic.bed.solids  = solid0
ic.bed.packing = random

ic.bed.solid0.volfrac  =  0.275
ic.bed.solid0.velocity =  0.00  0.00  0.00

ic.bed.solid0.diameter = constant
ic.bed.solid0.density  = constant

ic.bed.solid0.diameter.constant =  100.0e-6
ic.bed.solid0.density.constant  = 1000.0


#_______________________________________________________________________
# Boundary Conditions
#
bc.regions = inflow  outflow

bc.inflow = mi
bc.inflow.fluid.volfrac  =  1.0
bc.inflow.fluid.velocity =  0.015  0.0  0.0


bc.outflow = po
bc.outflow.fluid.pressure =  0.0

)");
  CHECK(maybe_st.has_value());
}
