#include "csg_cgal_helper.hpp"
#include "csg_exception.hpp"

#include <CGAL/AABB_face_graph_triangle_primitive.h>
#include <CGAL/AABB_traits.h>
#include <CGAL/AABB_tree.h>
#include <CGAL/Polygon_mesh_processing/triangulate_faces.h>
#include <CGAL/Polyhedron_incremental_builder_3.h>
#include <CGAL/algorithm.h>
#include <CGAL/boost/graph/graph_traits_Polyhedron_3.h>

namespace {

typedef cgal_helper::Polyhedron::HalfedgeDS HalfedgeDS;
typedef typename HalfedgeDS::Vertex Vertex;
typedef typename Vertex::Point Point;
typedef CGAL::Side_of_triangle_mesh<cgal_helper::Polyhedron, cgal_helper::CK>
    Point_inside;

// A modifier creating a polyhedron with the incremental builder.
template <class HDS> class PolyhedronBuilder : public CGAL::Modifier_base<HDS> {
private:
  const std::vector<std::tuple<double, double, double>> &m_points;
  const std::vector<std::vector<unsigned int>> &m_faces;

public:
  PolyhedronBuilder(
      const std::vector<std::tuple<double, double, double>> &points,
      const std::vector<std::vector<unsigned int>> &faces)
      : m_points(points), m_faces(faces) {}

  void operator()(HDS &hds) {
    CGAL::Polyhedron_incremental_builder_3<HDS> B(hds, true);
    B.begin_surface(m_points.size(), m_faces.size());

    // Add all the vertices first
    for (const auto &p : m_points) {
      auto [px, py, pz] = p;
      B.add_vertex(Point(px, py, pz));
    }

    // Add facets next
    for (const auto &face : m_faces) {
      B.begin_facet();
      if (!B.test_facet(face.begin(), face.end()))
        throw csg::Exception(
            "Polyhedron builder",
            "Unable to create the polyhedron. test_facet failed. Check input!");

      for (const auto &p_index : face) {
        B.add_vertex_to_facet(p_index);
      }
      B.end_facet();
    }

    B.end_surface();
  }
};

} // namespace

namespace cgal_helper {

std::shared_ptr<Polyhedron>
create_polyhedron(const std::vector<std::tuple<double, double, double>> &points,
                  const std::vector<std::vector<unsigned int>> &faces) {
  auto p = std::make_shared<Polyhedron>();

  // Build incrementally
  PolyhedronBuilder<HalfedgeDS> bp(points, faces);
  p->delegate(bp);
  CGAL_assertion(p->is_valid());

  // Triangulate faces - needed for levelset
  CGAL::Polygon_mesh_processing::triangulate_faces(*p);
  CGAL_assertion(p->is_valid());

  return p;
}

std::shared_ptr<AABBTree>
create_aabb_tree(const std::shared_ptr<Polyhedron> &polyhedron) {
  // Construct AABB tree with a KdTree
  auto tree = std::make_shared<AABBTree>(
      faces(*polyhedron).first, faces(*polyhedron).second, *polyhedron);
  tree->accelerate_distance_queries();
  return tree;
}

bool inside(const std::shared_ptr<AABBTree> &tree, double xx, double yy,
            double zz) {
  cgal_helper::CK::Point_3 pt(xx, yy, zz);
  Point_inside inside_tester(*tree);

  // Determine the side and return true if inside!
  return inside_tester(pt) == CGAL::ON_BOUNDED_SIDE;
}

} // namespace cgal_helper
